/*
 * Decompiled with CFR 0.152.
 */
package com.timestored.docs;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.timestored.docs.Document;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;
import java.util.regex.Pattern;

public class OpenDocumentsModel {
    private static final Logger LOG = Logger.getLogger(OpenDocumentsModel.class.getName());
    private final List<Document> documents = new CopyOnWriteArrayList<Document>();
    private Document selectedDocument;
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList();
    private Document.Listener selectedDocListener = new Document.Listener(){

        @Override
        public void docContentModified() {
            for (Document.Listener l : OpenDocumentsModel.this.listeners) {
                l.docContentModified();
            }
        }

        @Override
        public void docCaratModified() {
            for (Document.Listener l : OpenDocumentsModel.this.listeners) {
                l.docCaratModified();
            }
        }

        @Override
        public void docSaved() {
            for (Document.Listener l : OpenDocumentsModel.this.listeners) {
                l.docSaved();
            }
        }
    };
    private File selectedFolder;
    private Pattern ignoredFolderPattern;

    public static OpenDocumentsModel newInstance() {
        return new OpenDocumentsModel();
    }

    public void forceKeyboardShortcutOverrides() {
        KeyboardFocusManager.getCurrentKeyboardFocusManager().addKeyEventDispatcher(new KeyEventDispatcher(){
            boolean down = false;

            @Override
            public boolean dispatchKeyEvent(KeyEvent e) {
                int kc = e.getKeyCode();
                boolean ctrl = e.isControlDown();
                if (!this.down && ctrl && kc == 115) {
                    OpenDocumentsModel.this.closeDocument();
                    this.down = true;
                } else if (!this.down && ctrl && e.isShiftDown() && kc == 9) {
                    OpenDocumentsModel.this.gotoPrevDocument();
                    this.down = true;
                } else if (!this.down && ctrl && kc == 9) {
                    OpenDocumentsModel.this.gotoNextDocument();
                    this.down = true;
                } else {
                    this.down = false;
                }
                if (this.down) {
                    e.consume();
                }
                return false;
            }
        });
    }

    private OpenDocumentsModel() {
        this.selectedDocument = this.addDocument();
        this.selectedDocument.addListener(this.selectedDocListener);
    }

    public Document openDocument(String filepath) throws IOException {
        Preconditions.checkNotNull(filepath);
        return this.openDocument(new File(filepath));
    }

    public void openDocuments(List<String> filepaths) {
        for (String fp : filepaths) {
            try {
                this.openDocument(new File(fp));
            }
            catch (IOException e) {
                LOG.warning("Couldn't open previously opened file location:" + fp);
            }
        }
        if (this.documents.size() > 0) {
            this.changeSelectedDocTo(this.documents.get(this.documents.size() - 1));
        } else {
            this.changeSelectedDocTo(this.addDocument());
        }
        this.selectedDocument.addListener(this.selectedDocListener);
        for (Listener l : this.listeners) {
            l.docSelected(this.selectedDocument);
        }
    }

    public void changeSelectedDocTo(Document d) {
        if (this.selectedDocument != null) {
            this.selectedDocument.removeListener(this.selectedDocListener);
        }
        this.selectedDocument = d;
        this.selectedDocument.addListener(this.selectedDocListener);
    }

    public Document addDocument() {
        return this.addDocument(null);
    }

    public Document addDocument(String title) {
        Document d = new Document();
        if (title != null && title.trim().length() > 0) {
            d.setTitle(title);
        }
        LOG.info("addDocument: " + d.getTitle());
        this.documents.add(d);
        this.changeSelectedDocTo(d);
        for (Listener l : this.listeners) {
            l.docAdded(d);
            l.docSelected(d);
        }
        return d;
    }

    public Document openDocument(File file) throws IOException {
        LOG.info("openDocument: " + file.getName());
        for (Document d : this.documents) {
            if (d.getFilePath() == null || !d.getFilePath().equals(file.getAbsolutePath())) continue;
            LOG.info("openDocument: was already open, reselecting->" + file.getName());
            this.changeSelectedDocTo(d);
            for (Listener l : this.listeners) {
                l.docSelected(d);
            }
            return d;
        }
        Document d = new Document(file);
        this.documents.add(d);
        for (Listener l : this.listeners) {
            l.docAdded(d);
        }
        this.setSelectedDocument(d);
        return d;
    }

    public void closeDocument() {
        this.closeDocument(this.selectedDocument);
    }

    public void closeDocument(Document document) {
        boolean changedSelection;
        LOG.info("closeDocument: " + document.getTitle());
        boolean removed = this.documents.remove(document);
        boolean bl = changedSelection = this.selectedDocument == document;
        if (changedSelection) {
            if (this.documents.size() > 0) {
                this.changeSelectedDocTo(this.documents.get(0));
            } else {
                this.changeSelectedDocTo(this.addDocument());
            }
        }
        for (Listener l : this.listeners) {
            if (removed) {
                l.docClosed(document);
            }
            if (!changedSelection) continue;
            l.docSelected(this.selectedDocument);
        }
    }

    public void saveAs(File file, boolean useWindowsLineEndings) throws IOException {
        LOG.info("saveAs: " + this.selectedDocument.getTitle() + " as " + file.getAbsolutePath());
        this.selectedDocument.saveAs(file, useWindowsLineEndings);
    }

    public void saveDocument(boolean useWindowsLineEndings) throws IOException {
        LOG.info("saveDocument: " + this.selectedDocument.getTitle());
        this.selectedDocument.save(useWindowsLineEndings);
    }

    public void gotoNextDocument() {
        LOG.info("gotoNextDocument");
        int i = (this.documents.indexOf(this.selectedDocument) + 1) % this.documents.size();
        this.setSelectedDocument(this.documents.get(i));
    }

    public void gotoPrevDocument() {
        LOG.info("gotoPrevDocument");
        int i = this.documents.indexOf(this.selectedDocument) - 1;
        i = i < 0 ? this.documents.size() - 1 : i;
        this.setSelectedDocument(this.documents.get(i));
    }

    public void setSelectedDocument(Document document) {
        LOG.info("setSelectedDocument: " + document.getTitle());
        if (this.documents.contains(document)) {
            if (!document.equals(this.selectedDocument)) {
                this.changeSelectedDocTo(document);
            }
            for (Listener l : this.listeners) {
                l.docSelected(document);
            }
        } else {
            String msg = "I dont have doc: " + document.getTitle();
            LOG.warning(msg);
            throw new IllegalArgumentException(msg);
        }
    }

    public void setSelectedFolder(File selectedFolder) {
        boolean noChange;
        LOG.info("setSelectedFolder: " + selectedFolder);
        boolean bl = noChange = this.selectedFolder == selectedFolder || selectedFolder != null && selectedFolder.equals(this.selectedFolder);
        if (!noChange) {
            if (selectedFolder == null || selectedFolder.isDirectory()) {
                this.selectedFolder = selectedFolder;
                for (Listener l : this.listeners) {
                    l.folderSelected(selectedFolder);
                }
            } else {
                String msg = "not a directory: " + selectedFolder;
                LOG.warning(msg);
                throw new IllegalArgumentException(msg);
            }
        }
        this.selectedFolder = selectedFolder;
    }

    public void setContent(String content) {
        LOG.info("setContent carat=" + this.selectedDocument.getCaratPosition());
        this.selectedDocument.setContent(content);
    }

    public List<Document> getDocuments() {
        return this.documents;
    }

    public Document getSelectedDocument() {
        return this.selectedDocument;
    }

    public File getSelectedFolder() {
        return this.selectedFolder;
    }

    public boolean hasAnyUnsavedChanges() {
        for (Document d : this.documents) {
            if (!d.hasUnsavedChanges()) continue;
            return true;
        }
        return false;
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("documentsSize", this.documents.size()).add("selectedDocument", this.selectedDocument).add("listenersSize", this.listeners.size()).toString();
    }

    public void insertSelectedText(String text) {
        this.selectedDocument.insertSelectedText(text);
    }

    public void insertText(String text) {
        this.selectedDocument.insertText(text);
    }

    public void closeAll() {
        ArrayList<Document> docs = new ArrayList<Document>(this.documents);
        LOG.info("closeAll");
        this.documents.clear();
        this.changeSelectedDocTo(this.addDocument());
        for (Listener l : this.listeners) {
            l.docSelected(this.selectedDocument);
        }
        for (Document closedDoc : docs) {
            for (Listener l : this.listeners) {
                l.docClosed(closedDoc);
            }
        }
    }

    public void setIgnoredFoldersRegex(Pattern ignoredFolderPattern) {
        boolean noChange;
        LOG.info("setIgnoredFoldersRegex: " + ignoredFolderPattern);
        boolean bl = noChange = this.ignoredFolderPattern == ignoredFolderPattern || ignoredFolderPattern != null && ignoredFolderPattern.equals(this.ignoredFolderPattern);
        if (!noChange) {
            this.ignoredFolderPattern = ignoredFolderPattern;
            for (Listener l : this.listeners) {
                l.ignoredFolderPatternSelected(ignoredFolderPattern);
            }
        }
    }

    public void addListener(Listener listener) {
        if (!this.listeners.contains(listener)) {
            this.listeners.add(listener);
        }
    }

    public void removeListener(Listener listener) {
        this.listeners.remove(listener);
    }

    public Pattern getIgnoredFolderPattern() {
        return this.ignoredFolderPattern;
    }

    public static abstract class Adapter
    implements Listener {
        @Override
        public void docAdded(Document document) {
        }

        @Override
        public void docClosed(Document document) {
        }

        @Override
        public void docSelected(Document document) {
        }

        @Override
        public void docContentModified() {
        }

        @Override
        public void docCaratModified() {
        }

        @Override
        public void docSaved() {
        }

        @Override
        public void folderSelected(File selectedFolder) {
        }

        @Override
        public void ignoredFolderPatternSelected(Pattern ignoredFolderPattern) {
        }
    }

    public static interface Listener
    extends Document.Listener {
        public void docAdded(Document var1);

        public void docClosed(Document var1);

        public void docSelected(Document var1);

        public void folderSelected(File var1);

        public void ignoredFolderPatternSelected(Pattern var1);
    }
}

